<?php

namespace App\Filament\Pages;

use Filament\Pages\Page;
use App\Models\CashMovement;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\Filter;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Set;
use Filament\Forms\Get;
use Filament\Tables\Columns\TextColumn\TextColumnSize;
use Filament\Tables\Actions\ExportAction;
use App\Filament\Exports\CashMovementExporter;

class CashMovementsReport extends Page implements Tables\Contracts\HasTable
{
    use Tables\Concerns\InteractsWithTable;
    protected static ?string $navigationIcon = 'heroicon-o-document-currency-dollar';
    protected static ?string $navigationGroup = 'Reportes';
    protected static ?string $modelLabel = 'Movimientos de caja';
    protected static ?string $navigationLabel = 'Movimientos de caja ';
    protected static ?string $title = 'Movimientos de caja';
    protected static ?int $navigationSort = 2;

    protected static string $view = 'filament.pages.cash-movements-report';

    protected function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns($this->getTableColumns())
            ->filters($this->getTableFilters(),
                layout: FiltersLayout::AboveContent)
            ->filtersFormColumns(1)
            ->headerActions($this->getTableHeaderActions())
            ->defaultSort('created_at', 'desc')
            ->emptyStateIcon('heroicon-o-banknotes')
            ->emptyStateHeading('No hay movimientos de caja registrados')
            ->emptyStateDescription('No se encontraron movimientos de caja para mostrar.')
            ->emptyStateActions([
                Tables\Actions\Action::make('reload')
                    ->label('Recargar')
                    ->icon('heroicon-o-arrow-path')
                    ->action(fn() => $this->dispatch('reload')),
            ]);
    }

    protected function getTableQuery(): Builder
    {
        return CashMovement::query()
            ->with('user', 'reference')
            ->latest();
    }

    protected function getTableColumns(): array
    {
        return [
            TextColumn::make('reference_type')
                ->label('Origen')
                ->formatStateUsing(function ($state, $record) {
                    if (!$record->reference) {
                        return 'Sin origen';
                    }
                    return match ($record->reference_type) {
                        'App\Models\Appointment' => 'Cita del paciente: ' . $record->reference->patient->full_name,
                        //'App\Models\Purchase' => 'Compra #' . $record->reference->id,
                        default => class_basename($record->reference_type) . ' #' . $record->reference->id,
                    };
                }),
            TextColumn::make('user.full_name')
                ->label('Usuario'),
            TextColumn::make('type')
                ->label('Tipo')
                ->badge()
                ->color(fn(string $state) => $state === 'ingreso' ? 'success' : 'danger'),
            TextColumn::make('method')
                ->label('Metodo'),
            TextColumn::make('amount')
                ->label('Monto')
                ->numeric()
                ->money('BOB', true),
            TextColumn::make('created_at')
                ->label('Fecha')
                ->dateTime(),
        ];
    }
    protected function getTableFilters(): array
    {
        return [
            Filter::make('advanced_filter')
                ->form([
                    DatePicker::make('from')->label('Desde'),
                    DatePicker::make('until')->label('Hasta'),

                    Select::make('method')
                        ->label('Método de pago')
                        ->options([
                            'tarjeta' => 'Tarjeta',
                            'efectivo' => 'Efectivo',
                            'transferencia' => 'Transferencia',
                            'QR' => 'QR',
                        ])
                        ->placeholder('Selecciona metodo de pago'),

                    Select::make('type')
                        ->label('Tipo de movimiento')
                        ->options([
                            'ingreso' => 'Ingreso',
                            'egreso' => 'Egreso',
                        ])
                        ->placeholder('Selecciona tipo de movimiento'),
                ])
                ->columns(4)
                ->query(function ($query, array $data) {
                    return $query
                        ->when($data['from'], fn($q) => $q->whereDate('created_at', '>=', $data['from']))
                        ->when($data['until'], fn($q) => $q->whereDate('created_at', '<=', $data['until']))
                        ->when($data['method'], fn($q) => $q->where('method', $data['method']))
                        ->when($data['type'], fn($q) => $q->where('type', $data['type']));
                }),
        ];
    }
    protected function getTableHeaderActions(): array
    {
        return [
            ExportAction::make()
                ->exporter(CashMovementExporter::class)
                ->color('primary')
                ->label('Exportar a Excel')
                ->icon('heroicon-o-document-arrow-down'),
        ];
    }
    
    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()->hasRole('Administrador');
    }

    public static function canAccess(): bool
    {
        return auth()->user()->hasRole('Administrador');
    }
}
