<?php

namespace App\Filament\Pages;

use Filament\Pages\Page;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use App\Models\Appointment;
use Illuminate\Database\Eloquent\Model;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Actions\ExportAction;
use App\Filament\Exports\CommissionExporter;

class Commissions extends Page implements HasTable
{
    use Tables\Concerns\InteractsWithTable;

    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static string $view = 'filament.pages.commissions';
    protected static ?string $navigationGroup = 'Reportes';
    protected static ?int $navigationSort = 2;
    protected static ?string $title = 'Comisiones';
    protected static ?string $navigationLabel = 'Comisiones';

    public function table(Table $table): Table
    {
        return $table
            ->query(
                Appointment::query()
                    ->with('user', 'patient', 'doctor')
                    ->latest()
            )
            ->columns([
                Tables\Columns\TextColumn::make('patient.full_name')
                    ->label('Paciente')
                    ->searchable(
                        query: function (Builder $query, string $search): Builder {
                            return $query->whereHas('patient', function ($subQuery) use ($search) {
                                $subQuery->where('name', 'like', "%{$search}%")
                                    ->orWhere('last_name', 'like', "%{$search}%");
                            });
                        }
                    ),
                Tables\Columns\TextColumn::make('user.full_name')
                    ->label('Usuario')
                    ->searchable(
                        query: function (Builder $query, string $search): Builder {
                            return $query->whereHas('user', function ($subQuery) use ($search) {
                                $subQuery->where('name', 'like', "%{$search}%")
                                    ->orWhere('last_name', 'like', "%{$search}%");
                            });
                        }
                    ),
                Tables\Columns\TextColumn::make('doctor.full_name')
                    ->label('Medic@')
                    ->searchable(
                        query: function (Builder $query, string $search): Builder {
                            return $query->whereHas('doctor', function ($subQuery) use ($search) {
                                $subQuery->where('name', 'like', "%{$search}%")
                                    ->orWhere('last_name', 'like', "%{$search}%");
                            });
                        }
                    ),
                Tables\Columns\TextColumn::make('total_cost')
                    ->label('Costo total')
                    ->numeric()
                    ->money('BOB', true)
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->label('Estado')
                    ->badge()
                    ->color(fn(string $state) => match ($state) {
                        'pendiente' => 'warning',
                        'completado' => 'success',
                        'cancelado' => 'danger',
                        default => 'primary',
                    }),
                Tables\Columns\TextColumn::make('appointment_date')
                    ->label('Fecha de creación')
                    ->dateTime()
                    ->sortable(),
            ])
            ->headerActions([
                ExportAction::make()
                    ->exporter(CommissionExporter::class)
                    ->color('primary')
                    ->label('Exportar a Excel')
                    ->icon('heroicon-o-document-arrow-down'),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->options([
                        'pendiente' => 'Pendiente',
                        'completado' => 'Completado',
                        'cancelado' => 'Cancelado',
                    ])
                    ->label('Estado'),

                SelectFilter::make('doctor_id')
                    ->relationship('doctor', 'name', fn($query) => $query->role('Médico'))
                    ->getOptionLabelFromRecordUsing(fn(Model $record) => "{$record->name} {$record->last_name}")
                    ->searchable()
                    ->preload()
                    ->label('Médico'),

                Filter::make('appointment_date')
                    ->form([

                        DatePicker::make('from')->label('Desde'),
                        DatePicker::make('until')->label('Hasta'),
                    ])
                    ->columns(2)
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['from'], fn($q) => $q->whereDate('appointment_date', '>=', $data['from']))
                            ->when($data['until'], fn($q) => $q->whereDate('appointment_date', '<=', $data['until']));
                    })
                    ->label('Rango de fechas'),
            ], layout: FiltersLayout::AboveContent)
            ->filtersFormColumns(3);
    }
    
    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()->hasRole('Administrador');
    }

    public static function canAccess(): bool
    {
        return auth()->user()->hasRole('Administrador');
    }
}
