<?php

namespace App\Filament\Pages;

use Filament\Actions\Action;
use Filament\Pages\Page;
use Filament\Pages\Dashboard as BaseDashboard;
use Filament\Widgets;
use App\Models\CashRegister;


class Dashboard extends BaseDashboard
{
    public function getHeaderActions(): array
    {
        return [
            Action::make('openRegister')
                ->label('Abrir caja')
                ->icon('heroicon-o-banknotes')
                ->form([
                    \Filament\Forms\Components\TextInput::make('opening_amount')
                        ->label('Monto de apertura')
                        ->required()
                        ->numeric()
                        ->prefix('Bs'),
                ])
                ->action(function (array $data): void {
                    CashRegister::create([
                        'user_id' => auth()->id(),
                        'opened_at' => now(),
                        'opening_amount' => $data['opening_amount'],
                        'status' => 'abierto',
                    ]);
                })
                ->visible(function (): bool {
                    return !CashRegister::where('user_id', auth()->id())
                        ->where('status', 'abierto')
                        ->exists();
                })
                ->successNotificationTitle('Caja abierta correctamente'),

            // 👉 BOTÓN PARA CERRAR CAJA
            Action::make('closeRegister')
                ->label('Cerrar caja')
                ->icon('heroicon-m-lock-closed')
                ->form(function () {
                    $register = \App\Models\CashRegister::where('user_id', auth()->id())
                        ->where('status', 'abierto')
                        ->latest('opened_at')
                        ->first();

                    $transferencia = $register->movements->sum(function ($m) {
                        return $m->type === 'ingreso' && $m->method === 'transferencia' ? $m->amount : 0;
                    });
                    $tarjeta = $register->movements->sum(function ($m) {
                        return $m->type === 'ingreso' && $m->method === 'tarjeta' ? $m->amount : 0;
                    });
                    $qr = $register->movements->sum(function ($m) {
                        return $m->type === 'ingreso' && $m->method === 'QR' ? $m->amount : 0;
                    });
                    $efectivo = $register->movements->sum(function ($m) {
                        return $m->type === 'ingreso' && $m->method === 'efectivo' ? $m->amount : 0;
                    });
                    $total = $register->movements->sum(function ($m) {
                        return $m->type === 'ingreso' ? $m->amount : -$m->amount;
                    });

                    $totalSugerido = $register->opening_amount + $total;

                    return [
                        \Filament\Forms\Components\Placeholder::make('transferencia')
                            ->label('Pago por transferencias')
                            ->content("Bs {$transferencia}"),
                        \Filament\Forms\Components\Placeholder::make('tarjeta')
                            ->label('Pago con tarjetas')
                            ->content("Bs {$tarjeta}"),
                        \Filament\Forms\Components\Placeholder::make('qr')
                            ->label('Pago por QR')
                            ->content("Bs {$qr}"),
                        \Filament\Forms\Components\Placeholder::make('efectivo')
                            ->label('Pago con efectivo')
                            ->content("Bs {$efectivo}"),
                        \Filament\Forms\Components\TextInput::make('closing_amount')
                            ->label('Monto contado (cierre)')
                            ->required()
                            ->numeric()
                            ->prefix('Bs'),
                    ];
                })
                ->action(function (array $data): void {
                    $register = CashRegister::where('user_id', auth()->id())
                        ->where('status', 'abierto')
                        ->latest('opened_at')
                        ->first();

                    if ($register) {
                        $total = $register->movements->sum(function ($m) {
                            return $m->type === 'ingreso'
                                ? $m->amount
                                : -$m->amount;
                        });
                        $register->update([
                            'closed_at' => now(),
                            'status' => 'cerrado',
                            'total_amount' => $register->opening_amount + $total,
                            'closing_amount' => $data['closing_amount'],
                        ]);
                    }
                })
                ->visible(function (): bool {
                    return CashRegister::where('user_id', auth()->id())
                        ->where('status', 'abierto')
                        ->exists();
                })
                ->successNotificationTitle('Caja cerrada correctamente'),
        ];
    }

    function getWidgets(): array
    {
        return [
            Widgets\AccountWidget::class,
        ];
    }
}
