<?php

namespace App\Filament\Resources;

use App\Filament\Resources\AppointmentResource\Pages;
use App\Filament\Resources\AppointmentResource\RelationManagers;
use App\Models\Appointment;
use App\Models\TestPrice;
use App\Models\User;
use Illuminate\Database\Eloquent\Model;

use Dom\Text;
use Filament\Forms;
use Filament\Forms\Form;
use Spatie\Permission\Models\Role;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Fieldset;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Radio;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

use function Pest\Laravel\options;
use Barryvdh\DomPDF\Facade\Pdf;
use Filament\Forms\Components\Hidden;
use Illuminate\Support\Facades\URL;



class AppointmentResource extends Resource
{
    protected static ?string $model = Appointment::class;
    protected static ?string $modelLabel = 'Órden';
    protected static ?string $pluralModelLabel = 'Órdenes';
    protected static ?string $navigationGroup = 'Pacientes';
    protected static ?int $navigationSort = 2;
    protected static ?string $navigationLabel = 'Órdenes';


    protected static ?string $navigationIcon = 'heroicon-o-calendar-days';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('patient_id')
                    ->label('Paciente')
                    ->relationship('patient', 'name')
                    ->getOptionLabelFromRecordUsing(fn(Model $record) => "{$record->name} {$record->last_name}")
                    ->required()
                    ->preload(true)
                    ->searchable()
                    ->columnSpanFull()
                    ->createOptionForm([
                        TextInput::make('name')->required()->label('Nombre'),
                        TextInput::make('last_name')->required()->label('Apellido'),
                        TextInput::make('age_years')->label('Edad (Años)')->required()->numeric()->default('0'),
                        TextInput::make('age_months')->label('Edad (Meses)')->required()->numeric()->maxValue(11)->default('0'),
                        TextInput::make('age_days')->label('Edad (Días)')->required()->numeric()->maxValue(30)->default('0'),
                        Select::make('sex')->label('Sexo')->options(['masculino' => 'Masculino', 'femenino' => 'Femenino',])->required(),
                        TextInput::make('ci')->label('Cédula de identidad')->required()->maxLength(255),
                    ])
                    ->editOptionForm([
                        TextInput::make('name')->label('Nombre del Paciente')->disabled(),
                        TextInput::make('age_years')->label('Edad (Años)')->required()->numeric()->default('0'),
                        TextInput::make('age_months')->label('Edad (Meses)')->required()->numeric()->maxValue(11)->default('0'),
                        TextInput::make('age_days')->label('Edad (Días)')->required()->numeric()->maxValue(30)->default('0'),
                    ]),
                Select::make('doctor_id')
                    ->label('Médico')
                    ->relationship('doctor', 'name', fn($query) => $query->role('Médico'))
                    ->getOptionLabelFromRecordUsing(fn(Model $record) => "{$record->name} {$record->last_name}")
                    ->required()
                    ->preload(true)
                    ->searchable()
                    ->createOptionForm([
                        TextInput::make('name')->required()->label('Nombre'),
                        TextInput::make('last_name')->required()->label('Apellido'),
                    ])->createOptionUsing(function (array $data) {
                        $user = User::create($data);
                        $user->assignRole('Médico'); // Asignar rol directamente
                        return $user->getKey(); // Retorna el ID
                    }),
                DateTimePicker::make('appointment_date')
                    ->label('Fecha de la Cita')
                    ->default(now())
                    ->required(),
                Select::make('status')
                    ->label('Estado')
                    ->options([
                        'pendiente' => 'Pendiente',
                        'completado' => 'Completado',
                        'entregada' => 'Entregada',
                        'cancelada' => 'Cancelada',
                    ])
                    ->default('pendiente')
                    ->required(),
                TextInput::make('notes')
                    ->label('Notas')
                    ->maxLength(500),
                TextInput::make('total_cost')
                    ->label('Costo Total')
                    ->numeric()
                    ->default(0.00)
                    ->required()
                    ->readOnly(),
                Select::make('price_type')
                    ->label('Tipo de precio')
                    ->options([
                        'particular' => 'Particular',
                        'domicilio' => 'Domicilio',
                        'seguro' => 'Seguro',
                        'emergencia' => 'Emergencia',
                    ])
                    ->default('particular')
                    ->reactive()
                    ->required()

                    ->afterStateUpdated(function (callable $get, callable $set, $state) {
                        $testIds = $get('selected_test_ids') ?? [];
                        if (is_array($testIds)) {
                            $totalCost = TestPrice::whereIn('test_id', $testIds)
                                ->where('name', $state)
                                ->sum('price');

                            $set('total_cost', $totalCost);
                        } else {
                            $set('total_cost', 0.00);
                        }
                    }),
                TextInput::make('paid_amount')
                    ->label('Monto Pagado')
                    ->visibleOn('create')
                    ->numeric()
                    ->required(),
                Select::make('method')
                    ->label('Método de pago')
                    ->visibleOn('create')
                    ->options([
                        'efectivo' => 'Efectivo',
                        'tarjeta' => 'Tarjeta',
                        'transferencia' => 'Transferencia',
                        'QR' => 'QR',
                    ])
                    ->required(),
                Select::make('selected_test_ids')
                    ->label('Análisis o perfiles')
                    ->multiple()
                    ->preload()
                    ->options(
                        \App\Models\Test::where('is_active', true)
                            ->orderBy('name')
                            ->pluck('name', 'id')
                    )
                    ->required()
                    ->columnSpanFull()
                    ->reactive()
                    ->afterStateUpdated(function (callable $get, callable $set, $state) {
                        if (is_array($state)) {
                            $priceType = $get('price_type'); // leemos el tipo seleccionado
                            $totalCost = TestPrice::whereIn('test_id', $state)
                                ->where('name', $priceType)
                                ->sum('price');

                            $set('total_cost', $totalCost);
                        } else {
                            $set('total_cost', 0.00);
                        }
                    }),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->label('Código')
                    ->searchable(
                        query: fn(Builder $query, string $search) =>
                        $query->orWhere('appointments.id', 'like', "%{$search}%")
                    )
                    ->sortable(),
                TextColumn::make('patient.full_name')
                    ->label('Paciente')
                    ->searchable(
                        query: function (Builder $query, string $search): Builder {
                            return $query->whereHas('patient', function ($subQuery) use ($search) {
                                $subQuery->where('name', 'like', "%{$search}%")
                                    ->orWhere('last_name', 'like', "%{$search}%");
                            });
                        }
                    )
                    ->sortable(query: function (Builder $query, string $direction): Builder {
                        return $query->join('patients', 'appointments.patient_id', '=', 'patients.id')
                            ->orderBy('patients.last_name', $direction)
                            ->select('appointments.*'); // Es CRÍTICO volver a seleccionar las columnas originales
                    }),
                TextColumn::make('doctor.full_name')
                    ->label('Médico')
                    ->searchable(
                        query: function (Builder $query, string $search): Builder {
                            return $query->whereHas('doctor', function ($subQuery) use ($search) {
                                $subQuery->where('name', 'like', "%{$search}%")
                                    ->orWhere('last_name', 'like', "%{$search}%");
                            });
                        }
                    )
                    ->sortable(query: function (Builder $query, string $direction): Builder {
                        return $query->join('users', 'appointments.doctor_id', '=', 'users.id')
                            ->orderBy('users.last_name', $direction)
                            ->select('appointments.*'); // Es CRÍTICO volver a seleccionar las columnas originales
                    }),
                TextColumn::make('appointment_date')
                    ->label('Fecha de la Cita')
                    ->dateTime()
                    ->sortable(),
                TextColumn::make('status')
                    ->label('Estado')
                    ->sortable(),
                TextColumn::make('total_cost')
                    ->label('Costo')
                    ->numeric()
                    ->money('BOB', true)
                    ->sortable(),
                TextColumn::make('paid_amount')
                    ->label('Pagado')
                    ->numeric()
                    ->money('BOB', true)
                    ->sortable(),
                TextColumn::make(('balance_due'))
                    ->label('Pendiente')
                    ->numeric()
                    ->money('BOB', true)
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Action::make('results')
                    ->label('Resultados')
                    ->tooltip('Ver resultados de los análisis')
                    ->url(fn($record) => static::getUrl('results', ['record' => $record->id]))
                    ->icon('heroicon-o-document-text'),
                Action::make('Exportar PDF')
                    ->label('Exportar PDF')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->url(fn($record) => route('appointments.export.pdf', $record))
                    ->openUrlInNewTab(),
                Action::make('payments')
                    ->label('Agregar pago')
                    ->icon('heroicon-m-currency-dollar')
                    ->visible(fn($record) => $record->balance_due > 0)
                    ->form([
                        TextInput::make('new_amount')
                            ->label('Monto del pago')
                            ->required()
                            ->numeric()
                            ->maxValue(fn($record) => $record->balance_due)
                            ->prefix('Bs'),
                        Select::make('new_method')
                            ->label('Método de pago')
                            ->options([
                                'efectivo' => 'Efectivo',
                                'tarjeta' => 'Tarjeta',
                                'transferencia' => 'Transferencia',
                                'QR' => 'QR',
                            ])
                            ->required(),
                    ])
                    ->action(function ($record, array $data) {
                        $user = auth()->user();

                        $openCashRegister = \App\Models\CashRegister::where('user_id', $user->id)
                            ->where('status', 'abierto')
                            ->latest('opened_at')
                            ->first();

                        if (! $openCashRegister) {
                            throw new \Exception('Debe abrir una caja para registrar el pago.');
                        }

                        // Registrar el movimiento de caja
                        \App\Models\CashMovement::create([
                            'cash_register_id' => $openCashRegister->id,
                            'user_id' => $user->id,
                            'type' => 'ingreso',
                            'method' => $data['new_method'], // puedes hacerlo dinámico
                            'amount' => $data['new_amount'],
                            'reference_id' => $record->id,
                            'reference_type' => \App\Models\Appointment::class,
                        ]);

                        // Actualizar el monto pagado
                        $record->increment('paid_amount', $data['new_amount']);
                        $record->decrement('balance_due', $data['new_amount']);

                        // Cambiar estado del pago si ya está completo
                        /*if ($record->paid_amount >= $record->total_amount) {
                            $record->update(['payment_status' => 'pagado']);
                        }*/
                    })
                    ->successNotificationTitle('Pago registrado correctamente')

            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAppointments::route('/'),
            'create' => Pages\CreateAppointment::route('/create'),
            'edit' => Pages\EditAppointment::route('/{record}/edit'),
            'view' => Pages\ViewAppointment::route('/{record}'),
            'results' => Pages\ManageResults::route('/{record}/results'),
        ];
    }

    public static function generateResultsFromAppointment($appointmentId): array
    {
        $appointment = \App\Models\Appointment::with(['patient', 'appointmentTests.test.components'])->find($appointmentId);

        if (! $appointment) return [];

        $patient = $appointment->patient;
        $sex = $patient->sex;
        $years = $patient->age_years ?? 0;
        $months = $patient->age_months ?? 0;
        $days = $patient->age_days ?? 0;

        $components = collect();

        foreach ($appointment->appointmentTests as $appointmentTest) {
            $test = $appointmentTest->test;
            $testName = $test->name;
            if (!$test || !$test->components) continue;

            $realComponents = $test->components->where('is_group', false);

            foreach ($realComponents as $component) {
                $group = $test->components->firstWhere('id', $component->parent_id);

                // Rango por edad y sexo
                /* $range = \App\Models\ComponentResultRange::where('component_id', $component->id)
                    ->where(function ($query) use ($sex) {
                        $query->whereNull('sex')
                            ->orWhere('sex', $sex)
                            ->orWhere('sex', 'ambos');
                    })
                    ->where(function ($query) use ($years, $months, $days) {
                        $query->where(function ($q) use ($years, $months, $days) {
                            $q->where('age_from_years', '<=', $years)
                                ->where('age_from_months', '<=', $months)
                                ->where('age_from_days', '<=', $days);
                        })
                            ->where(function ($q) use ($years, $months, $days) {
                                $q->where('age_to_years', '>=', $years)
                                    ->where('age_to_months', '>=', $months)
                                    ->where('age_to_days', '>=', $days);
                            });
                    })
                    ->first();*/

                // Opciones si existen
                $options = $component->resultOptions()->pluck('result_option', 'result_option')->toArray();

                $components->push([
                    'appointment_test_id' => $appointmentTest->id,
                    'test_component_id' => $component->id,
                    'component_name' => $component->name,
                    'unit' => $component->unit,
                    'key' => $component->key,
                    'is_calculated' => $component->is_calculated,
                    'calculation_formula' => $component->calculation_formula,
                    'calculation_dependencies' => $component->calculation_dependencies,
                    //'reference_range' => $range ? "{$range->min_value} - {$range->max_value} {$range->unit}" : null,
                    'input_value' => '',
                    'option_value' => null,
                    'group_name' => $group?->name,
                    'options' => $options,
                    'test_name' => $testName,
                ]);
            }
        }

        return $components->toArray();
    }
}
