<?php

namespace App\Filament\Resources\AppointmentResource\Pages;

use App\Filament\Resources\AppointmentResource;
use Filament\Resources\Pages\Page;
use Filament\Notifications\Notification;
use Illuminate\Support\Collection;
use App\Models\TestObservation;
use App\Models\AppointmentTest;
use App\Models\TestCategory;
use App\Models\TestComponentResult;
use Filament\Actions;

class ManageResults extends Page
{
    protected static string $resource = AppointmentResource::class;
    protected static string $view = 'filament.resources.appointment-resource.pages.manage-results';
    protected static ?string $title = 'Resultados de Exámenes';

    // Propiedades públicas (asegúrate que estén públicas)
    public $saved = false;
    public $appointment;
    public Collection $resultsData;
    public $formData = [];
    public $observations = []; // ['category_1' => 'texto', 'test_23' => 'texto', ...]
    public $appointmentTestInfo = [];

    // Modal control
    public $activeGroupKey = null;
    public $showModal = false;

    // Mapa para recálculos: dependencyIndex => [calculatedIndex,...]
    public $dependencyToCalculatedMap = [];

    // Mapa de cálculo: calcIndex => ['formula'=>..., 'dependencies'=>[...] ]
    public $calculationMap = [];

    // Mapa key => index (útil para cálculos cliente)
    public $keyToIndex = [];

    public function mount($record): void
    {
        $this->appointment = AppointmentResource::getModel()::with([
            'appointmentTests.test.components',
            'testComponentResults',
        ])->findOrFail($record);

        foreach ($this->appointment->appointmentTests as $apptTest) {
            $this->appointmentTestInfo[$apptTest->id] = [
                'is_grouped' => (bool) ($apptTest->test->is_grouped ?? false),
                'category_id' => $apptTest->test->category->id ?? null,
                'category_name' => $apptTest->test->category->name ?? null,
                'test_name' => $apptTest->test->name ?? $apptTest->id,
            ];
        }

        $this->resultsData = collect(AppointmentResource::generateResultsFromAppointment($this->appointment->id));
        $this->formData = [];
        $this->observations = [];

        $index = 0;

        foreach ($this->resultsData as $component) {
            $existingResult = $this->appointment->testComponentResults()
                ->where('component_id', $component['test_component_id'])
                ->where('appointment_test_id', $component['appointment_test_id'])
                ->first();

            $this->formData[$index] = [
                'input_value' => $existingResult->input_value ?? '',
                'option_value' => $existingResult->option_value ?? '',
                'test_component_id' => $component['test_component_id'],
                'appointment_test_id' => $component['appointment_test_id'],
                'component_name' => $component['component_name'],
                'key' => $component['key'],
                'is_calculated' => $component['is_calculated'] ?? false,
                'calculation_formula' => $component['calculation_formula'] ?? null,
                'calculation_dependencies' => $component['calculation_dependencies'] ?? [],
                'unit' => $component['unit'] ?? null,
                'options' => $component['options'] ?? [],
                'group_name' => $component['group_name'] ?? null,
                'test_name' => $component['test_name'] ?? null,
                'test_is_grouped' => $component['test_is_grouped'] ?? ($this->appointmentTestInfo[$component['appointment_test_id']]['is_grouped'] ?? false),
                'test_category_id' => $component['test_category_id'] ?? ($this->appointmentTestInfo[$component['appointment_test_id']]['category_id'] ?? null),
            ];

            // guardar key->index para uso futuro (client-side)
            if (!empty($component['key'])) {
                $this->keyToIndex[$component['key']] = $index;
            }

            $index++;
        }

        // Observaciones iniciales (por test o categoría)
        $seenKeys = [];
        foreach ($this->formData as $c) {
            $apptTestId = $c['appointment_test_id'];
            $info = $this->appointmentTestInfo[$apptTestId] ?? null;

            if ($info && $info['is_grouped'] && $info['category_id']) {
                $key = 'category_'.$info['category_id'];
                if (!isset($seenKeys[$key])) {
                    $seenKeys[$key] = true;
                    $obs = TestObservation::where('appointment_id', $this->appointment->id)
                        ->where(function($q) use ($info) {
                            $q->where('observable_type', 'category')
                              ->orWhere('observable_type', TestCategory::class);
                        })
                        ->where('observable_id', $info['category_id'])
                        ->first();
                    $this->observations[$key] = $obs->observation ?? '';
                }
            } else {
                $key = 'test_'.$apptTestId;
                if (!isset($seenKeys[$key])) {
                    $seenKeys[$key] = true;
                    $obs = TestObservation::where('appointment_id', $this->appointment->id)
                        ->where(function($q) {
                            $q->where('observable_type', 'test')
                              ->orWhere('observable_type', AppointmentTest::class);
                        })
                        ->where('observable_id', $apptTestId)
                        ->first();
                    $this->observations[$key] = $obs->observation ?? '';
                }
            }
        }

        // Construir mapa de dependencias para recálculos y calculationMap
        $calculationMapLocal = [];
        foreach ($this->formData as $idx => $comp) {
            if (!empty($comp['is_calculated']) && !empty($comp['calculation_dependencies'])) {
                $calculationMapLocal[$idx] = [
                    'dependencies' => $comp['calculation_dependencies'],
                    'formula' => $comp['calculation_formula'],
                ];
            }
        }

        $this->calculationMap = $calculationMapLocal;

        $this->dependencyToCalculatedMap = [];
        foreach ($calculationMapLocal as $calcIndex => $calcData) {
            foreach ($calcData['dependencies'] as $depKey) {
                foreach ($this->formData as $depIndex => $comp) {
                    if (($comp['key'] ?? null) === $depKey) {
                        $this->dependencyToCalculatedMap[$depIndex][] = $calcIndex;
                    }
                }
            }
        }
    }

    // Construye estructura grupos (category_# o test_#) => ['label'=>, 'items'=>[[index,component],...]]
    public function buildGroupsStructure(): array
    {
        $fieldsForGroup = [];

        foreach ($this->formData as $index => $component) {
            $apptTestId = $component['appointment_test_id'];
            $info = $this->appointmentTestInfo[$apptTestId] ?? null;

            if ($info && $info['is_grouped'] && $info['category_id']) {
                $groupKey = 'category_' . $info['category_id'];
                $groupLabel = $info['category_name'] ?? ('Categoría '.$info['category_id']);
            } else {
                $groupKey = 'test_' . $apptTestId;
                $groupLabel = $info['test_name'] ?? $component['test_name'] ?? "Test {$apptTestId}";
            }

            if (!isset($fieldsForGroup[$groupKey])) {
                $fieldsForGroup[$groupKey] = [
                    'label' => $groupLabel,
                    'items' => [],
                ];
            }

            $fieldsForGroup[$groupKey]['items'][] = [
                'index' => $index,
                'component' => $component,
            ];
        }

        return $fieldsForGroup;
    }

    // Abre modal para un grupo
    public function openGroup(string $groupKey)
    {
        $this->activeGroupKey = $groupKey;
        $this->showModal = true;
    }

    // Cierra modal
    public function closeModal()
    {
        $this->activeGroupKey = null;
        $this->showModal = false;
    }

    // Recalcula los campos dependientes cuando cambia un input (se llama desde blade)
    /*public function recalcField(int $changedIndex)
    {
        if (!isset($this->dependencyToCalculatedMap[$changedIndex])) return;

        foreach ($this->dependencyToCalculatedMap[$changedIndex] as $targetIndex) {
            $target = $this->formData[$targetIndex] ?? null;
            if (!$target) continue;

            $formula = $target['calculation_formula'] ?? '';
            $dependencies = $target['calculation_dependencies'] ?? [];
            $evaluated = $formula;

            foreach ($dependencies as $depKey) {
                $value = 0;
                foreach ($this->formData as $i => $comp) {
                    if (($comp['key'] ?? null) === $depKey) {
                        $val = $this->formData[$i]['input_value'] ?? $this->formData[$i]['option_value'] ?? 0;
                        $value = is_numeric($val) ? $val : floatval(preg_replace('/[^0-9.\-]/','',$val));
                        break;
                    }
                }
                $evaluated = str_replace("{{$depKey}}", $value, $evaluated);
            }

            try {
                $result = eval("return $evaluated;");
                $this->formData[$targetIndex]['input_value'] = is_numeric($result) ? round($result, 1) : $result;
            } catch (\Throwable $e) {
                // ignorar errores de evaluación
            }
        }
    }*/
    
    
    public function recalcField(int $changedIndex)
{
    // seguridad: si no hay mapeo inicial, no hacemos nada
    if (empty($this->dependencyToCalculatedMap) || !isset($this->dependencyToCalculatedMap[$changedIndex])) {
        return;
    }

    // inicializamos cola con los calculados que dependen del campo cambiado
    $queue = $this->dependencyToCalculatedMap[$changedIndex] ?? [];
    if (!is_array($queue)) $queue = [$queue];

    $visitedCount = [];
    $maxVisitsPerIndex = max(10, count($this->formData) * 3);

    while (!empty($queue)) {
        $targetIndex = array_shift($queue);

        // comprobar existencia del target index
        if (!isset($this->formData[$targetIndex])) {
            // registrar y continuar
            logger()->warning("recalcField: targetIndex no existe", ['targetIndex' => $targetIndex, 'changedIndex' => $changedIndex]);
            continue;
        }

        // proteger contra loops: contamos visitas
        if (!isset($visitedCount[$targetIndex])) $visitedCount[$targetIndex] = 0;
        $visitedCount[$targetIndex]++;
        if ($visitedCount[$targetIndex] > $maxVisitsPerIndex) {
            logger()->warning("recalcField: max visits alcanzado para index, posible ciclo", ['index' => $targetIndex]);
            continue;
        }

        $target = $this->formData[$targetIndex];
        $formula = $target['calculation_formula'] ?? null;
        $dependencies = $target['calculation_dependencies'] ?? [];

        if (empty($formula) || empty($dependencies)) {
            // nada que calcular
            continue;
        }

        // construir expresión reemplazando placeholders
        $evaluated = $formula;
        $missing = false;
        foreach ($dependencies as $depKey) {
            $depIndex = $this->keyToIndex[$depKey] ?? null;

            // fallback: buscar por key directamente si keyToIndex no tiene mapeo
            if ($depIndex === null) {
                foreach ($this->formData as $i => $c) {
                    if (($c['key'] ?? null) === $depKey) {
                        $depIndex = $i;
                        break;
                    }
                }
            }

            if ($depIndex === null || !isset($this->formData[$depIndex])) {
                // no tenemos valor aún: marcamos missing y saltamos esta fórmula por ahora
                $missing = true;
                logger()->info("recalcField: dependencia no encontrada - se pospone cálculo", ['targetIndex' => $targetIndex, 'depKey' => $depKey]);
                break;
            }

            $num = $this->getNumericValueFromIndex($depIndex);
            if ($num === null) {
                // dependencia sin valor numérico -> posponer
                $missing = true;
                logger()->info("recalcField: dependencia sin valor numérico - se pospone cálculo", ['targetIndex' => $targetIndex, 'depKey' => $depKey, 'depIndex' => $depIndex]);
                break;
            }

            // sustituir todas las ocurrencias
            $evaluated = str_replace("{{$depKey}}", (string)$num, $evaluated);
        }

        if ($missing) {
            // saltamos evaluación, pero no descartamos; si más adelante se llena la dependencia, se reenviará desde esa dependencia
            continue;
        }

        // validaciones finales: no evaluar cadenas vacías
        if (trim($evaluated) === '') {
            logger()->warning("recalcField: expresión evaluada vacía", ['targetIndex' => $targetIndex, 'formula' => $formula]);
            continue;
        }

        $result = $this->safeEval($evaluated);
        if ($result === null) {
            // error en evaluación: lo registramos y mostramos alerta ligera
            logger()->error("recalcField: safeEval devolvió null", ['expr' => $evaluated, 'targetIndex' => $targetIndex]);
            Notification::make()
                ->title("Error de cálculo")
                ->danger()
                ->body("No se pudo evaluar la fórmula para el componente: {$target['component_name']} (index {$targetIndex}). Revisa la fórmula.")
                ->send();
            continue;
        }

        // formateo: redondeo a 1 decimal (igual que tenías)
        $newVal = is_numeric($result) ? round($result, 1) : $result;
        $oldVal = $this->formData[$targetIndex]['input_value'] ?? null;

        if ($oldVal != $newVal) {
            $this->formData[$targetIndex]['input_value'] = $newVal;

            // encolar dependientes de este target para propagar en cadena
            if (!empty($this->dependencyToCalculatedMap[$targetIndex])) {
                $deps = $this->dependencyToCalculatedMap[$targetIndex];
                if (!is_array($deps)) $deps = [$deps];
                foreach ($deps as $d) {
                    $queue[] = $d;
                }
            }
        }
    }
}

/**
 * Extrae un valor numérico limpio desde formData[index].
 * Retorna float o null si no hay número válido.
 */
protected function getNumericValueFromIndex(int $index)
{
    if (!isset($this->formData[$index])) return null;

    $comp = $this->formData[$index];
    $raw = $comp['input_value'] ?? ($comp['option_value'] ?? null);

    if ($raw === null || $raw === '') return null;

    // limpiar NBSP y espacios
    if (is_string($raw)) {
        $raw = str_replace(["\xC2\xA0", ' '], '', $raw);
        $raw = str_replace(',', '.', $raw);
    }

    // si ya es numérico
    if (is_numeric($raw)) return (float)$raw;

    // intentar extraer primer número con regex
    if (preg_match('/-?[0-9]+(?:\.[0-9]+)?/', (string)$raw, $m)) {
        return (float)$m[0];
    }

    return null;
}

/**
 * Evalúa una expresión matemática simple de forma controlada.
 * Devuelve número o null si inválida.
 */
protected function safeEval(string $expr)
{
    $expr = trim($expr);

    // sólo permitir dígitos, operadores +-*/() y punto decimal y espacios
    if (preg_match('/[^0-9\.\+\-\*\/\(\)\s]/', $expr)) {
        logger()->error("safeEval: caracteres no permitidos en expresión", ['expr' => $expr]);
        return null;
    }

    // evitar secuencias problemáticas
    if (strpos($expr, '//') !== false || strpos($expr, '++') !== false) {
        logger()->error("safeEval: secuencias prohibidas en expresión", ['expr' => $expr]);
        return null;
    }

    try {
        $res = @eval('return ' . $expr . ';');
        if (!is_numeric($res)) {
            logger()->error("safeEval: resultado no numérico", ['expr' => $expr, 'res' => $res]);
            return null;
        }
        return $res;
    } catch (\Throwable $e) {
        logger()->error("safeEval: excepción eval", ['expr' => $expr, 'exception' => $e->getMessage()]);
        return null;
    }
}

    
    
    

    // Guarda solo los componentes del grupo activo y su observación
    // Dentro de tu clase ManageResults reemplaza la función saveGroup por esta:

public function saveGroup()
{
    $groupKey = $this->activeGroupKey;
    if (!$groupKey) return;

    // Determinar si es categoría
    if (str_starts_with($groupKey, 'category_')) {
        $categoryId = (int) str_replace('category_', '', $groupKey);
        $isCategory = true;
    } else {
        $isCategory = false;
        $apptTestId = (int) str_replace('test_', '', $groupKey);
    }

    // Guardar componentes del grupo
    foreach ($this->formData as $component) {
        $apptTestIdOfComponent = $component['appointment_test_id'];
        $info = $this->appointmentTestInfo[$apptTestIdOfComponent] ?? null;

        $belongsToGroup = false;
        if ($isCategory) {
            if (($info['is_grouped'] ?? false) && ($info['category_id'] ?? null) === $categoryId) {
                $belongsToGroup = true;
            }
        } else {
            if ($apptTestIdOfComponent === $apptTestId) {
                $belongsToGroup = true;
            }
        }

        if (!$belongsToGroup) continue;

        \App\Models\TestComponentResult::updateOrCreate(
            [
                'appointment_test_id' => $component['appointment_test_id'],
                'component_id' => $component['test_component_id'],
            ],
            [
                'input_value' => $component['input_value'] ?? null,
                'option_value' => $component['option_value'] ?? null,
                'unit' => $component['unit'] ?? null,
            ]
        );
    }

    // Observación: guardar si hay texto; si está vacía, eliminar registro si existe
    $obsText = $this->observations[$groupKey] ?? null;
    $obsTrim = is_string($obsText) ? trim($obsText) : null;

    if ($obsTrim !== null && $obsTrim !== '') {
        // Guardar o actualizar
        if ($isCategory) {
            TestObservation::updateOrCreate(
                [
                    'appointment_id' => $this->appointment->id,
                    'observable_type' => 'category',
                    'observable_id' => $categoryId,
                ],
                ['observation' => $obsTrim]
            );
        } else {
            TestObservation::updateOrCreate(
                [
                    'appointment_id' => $this->appointment->id,
                    'observable_type' => 'test',
                    'observable_id' => $apptTestId,
                ],
                ['observation' => $obsTrim]
            );
        }
    } else {
        // Si está vacío: eliminar cualquier registro previo en DB para este observable
        if ($isCategory) {
            TestObservation::where('appointment_id', $this->appointment->id)
                ->where(function($q) {
                    $q->where('observable_type', 'category')
                      ->orWhere('observable_type', \App\Models\TestCategory::class);
                })
                ->where('observable_id', $categoryId)
                ->delete();
        } else {
            TestObservation::where('appointment_id', $this->appointment->id)
                ->where(function($q) {
                    $q->where('observable_type', 'test')
                      ->orWhere('observable_type', \App\Models\AppointmentTest::class);
                })
                ->where('observable_id', $apptTestId)
                ->delete();
        }

        // Actualizar estado local a string vacío para que la UI refleje que no hay observación
        $this->observations[$groupKey] = '';
    }

    Notification::make()
        ->title("Resultados guardados")
        ->success()
        ->send();

    $this->showModal = false;
    $this->activeGroupKey = null;
    $this->saved = true;
}

protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('export_pdf')
                ->label('Exportar PDF')
                ->icon('heroicon-o-arrow-down-tray')
                // pasar el modelo permite que la ruta lo reciba por model binding
                ->url(fn() => route('appointments.export.pdf', $this->appointment))
                ->openUrlInNewTab(),
        ];
    }
}
