<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PurchaseResource\Pages;
use App\Filament\Resources\PurchaseResource\RelationManagers;
use App\Models\Purchase;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Livewire\Attributes\Reactive;

class PurchaseResource extends Resource
{
    protected static ?string $model = Purchase::class;

    protected static ?string $navigationIcon = 'heroicon-o-arrow-path-rounded-square';
    protected static ?string $navigationGroup = 'Inventario';
    protected static ?string $modelLabel = 'Compra';
    protected static ?string $pluralModelLabel = 'Compras';
    protected static ?string $navigationLabel = 'Compras';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('supplier_name')
                    ->label('Nombre del proveedor')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('reference_number')
                    ->label('Número de referencia')
                    ->maxLength(255)
                    ->default(null),
                Forms\Components\DatePicker::make('date')
                    ->label('Fecha de compra')
                    ->default(now())
                    ->required(),
                Forms\Components\TextInput::make('total_amount')
                    ->label('Monto total')
                    ->default(0)
                    ->required()
                    ->reactive()
                    ->readOnly()
                    ->dehydrated(true)
                    ->numeric(),
                Forms\Components\Select::make('product_selector')
                    ->label('Buscar producto')
                    ->options(
                        \App\Models\ProductPresentation::with('product')
                            ->get()
                            ->mapWithKeys(function ($presentation) {
                                return [
                                    $presentation->id => $presentation->product->name . ' - ' . $presentation->label
                                ];
                            })
                            ->toArray()
                    )
                    ->searchable()
                    ->live()
                    ->columnSpanFull()
                    ->afterStateUpdated(function ($state, Set $set, Get $get) {
                        $currentProducts = $get('products') ?? [];
                        // Verifica si ya está agregado ese product_presentation_id
                        if (collect($currentProducts)->pluck('product_presentation_id')->contains($state)) {
                            return;
                        }
                        // Busca la presentación con el ID seleccionado
                        $presentation = \App\Models\ProductPresentation::with('product')->find($state);

                        if ($presentation && $presentation->product) {
                            $set('products', array_merge($currentProducts, [
                                [
                                    'product_presentation_id' => $presentation->id,
                                    'presentation' => $presentation->quantity,
                                    'product_name' => $presentation->product->name . ' - ' . $presentation->label,
                                    'unit_cost' => 0,
                                    'lot_number' => null,
                                    'expiration_date' => null,
                                    'quantity' => 1,
                                ],
                            ]));
                        }
                    }),
                Forms\Components\Repeater::make('products')
                    ->label('Productos')
                    ->columnSpanFull()
                    ->defaultItems(0)
                    ->addable(false)
                    ->reorderable(false)
                    ->columns(4)
                    ->deletable(false)
                    ->itemLabel(function (array $state): ?string {
                        return $state['product_name'] ?? 'Producto';
                    })
                    ->reactive()
                    ->afterStateHydrated(function (callable $set, $state, $record) {
                        if (!$record) return;

                        $set('products', $record->purchaseProducts->map(function ($item) {
                            return [
                                'purchase_product_id' => $item->id,
                                'product_presentation_id' => $item->product_presentation_id,
                                'presentation' => $item->productPresentation->quantity,
                                'product_name' => $item->productPresentation->product->name . ' - ' . $item->productPresentation->label,
                                'quantity' => $item->quantity,
                                'unit_cost' => $item->unit_cost,
                                'lot_number' => $item->lot_number,
                                'expiration_date' => $item->expiration_date,
                            ];
                        })->toArray());
                    })
                    ->schema([
                        Forms\Components\Hidden::make('purchase_product_id'), // para saber si es edición
                        Forms\Components\Hidden::make('product_presentation_id'),
                        Forms\Components\Hidden::make('presentation'),
                        Forms\Components\TextInput::make('quantity')
                            ->label('Cantidad')
                            ->required()
                            ->numeric()
                            ->default(1),
                        Forms\Components\TextInput::make('unit_cost')
                            ->label('Precio')
                            ->required()
                            ->live(onBlur: true)
                            ->numeric(),
                        Forms\Components\TextInput::make('lot_number')
                            ->label('Número de lote')
                            ->required()
                            ->maxLength(255)
                            ->default(null),
                        Forms\Components\DatePicker::make('expiration_date')
                            ->required()
                            ->label('Fecha de caducidad')

                    ])
                    ->afterStateUpdated(function ($state, \Filament\Forms\Set $set) {
                        $total = collect($state)->sum(function ($item) {
                            return ($item['unit_cost'] ?? 0);
                        });

                        $set('total_amount', $total);
                    }),
            ]);
        //->model(Purchase::with('purchaseProducts.productPresentation.product'));
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('supplier_name')
                    ->label('Proveedor')
                    ->searchable(),
                Tables\Columns\TextColumn::make('reference_number')
                    ->label('Número de referencia')
                    ->searchable(),
                Tables\Columns\TextColumn::make('date')
                    ->label('Fecha de compra')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_amount')
                    ->label('Monto total')
                    ->numeric()
                    ->money('BOB', true)
                    ->sortable(),
                Tables\Columns\TextColumn::make('user.full_name')
                    ->label('Usuario')
                    ->numeric()
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->recordUrl(null)
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPurchases::route('/'),
            'create' => Pages\CreatePurchase::route('/create'),
            'edit' => Pages\EditPurchase::route('/{record}/edit'),
        ];
    }
}
