<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TestComponentResource\Pages;
use App\Filament\Resources\TestComponentResource\RelationManagers;
use App\Models\TestComponent;
use Dom\Text;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\TextInputColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Grouping\Group;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class TestComponentResource extends Resource
{
    protected static ?string $model = TestComponent::class;
    protected static ?string $modelLabel = 'Componente de Análisis';
    protected static ?string $pluralModelLabel = 'Componentes de Análisis';
    protected static ?string $navigationGroup = 'Laboratorio';
    protected static ?string $navigationLabel = 'Componentes de Análisis';
    protected static ?int $navigationSort = 2;

    protected static ?string $navigationIcon = 'heroicon-o-eye-dropper';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Grid::make(2)->schema([
                    Select::make('test_id')
                        ->label('Análisis')
                        ->relationship('test', 'name')
                        ->searchable()
                        ->preload()
                        ->reactive()
                        ->required()
                        ->visibleOn('create'),
                    TextInput::make('name')
                        ->required()
                        ->label('Nombre del componente')
                        ->live(onBlur: true)
                        ->afterStateUpdated(function (Set $set, $state) {
                            $set('key', str($state)->slug());
                        }),

                    TextInput::make('key')
                        ->required()
                        ->dehydrated(true)
                        ->unique(TestComponent::class, 'key', ignoreRecord: true)
                        ->label('Clave del componente (usada en fórmulas)'),
                    Checkbox::make('is_group')
                        ->default(false)
                        ->visible(false)
                        ->dehydrated(true),
                    Select::make('result_type')
                        ->options([
                            'number' => 'Numérico',
                            'text' => 'Texto',
                        ])
                        ->label('Tipo de resultado'),
                    Select::make('parent_id')
                        ->label('Componente padre')
                        ->disabled(fn(Get $get) => !$get('test_id'))
                        ->options(function (Get $get) {
                            $testId = $get('test_id');

                            if (!$testId) {
                                return [];
                            }

                            return \App\Models\TestComponent::where('test_id', $testId)
                                ->where('is_group', true)
                                ->pluck('name', 'id');
                        })
                        ->createOptionForm([
                            Hidden::make('is_group')->default(true),
                            TextInput::make('name_group')
                                ->label('Nombre del componente padre')
                                ->required()
                                ->maxLength(255)
                                ->live(onBlur: true)
                                ->afterStateUpdated(function (Set $set, $state) {
                                    $set('key_group', str($state)->slug());
                                }),
                            TextInput::make('key_group')
                                ->label('Clave del componente padre (usada en fórmulas)')
                                ->required()
                                ->unique(TestComponent::class, 'key', ignoreRecord: true),
                        ])
                        ->createOptionUsing(function (array $data) {
                            $data['name'] = $data['name_group'];
                            $data['key'] = $data['key_group'];

                            return \App\Models\TestComponent::create($data)->getKey();
                        })
                        ->createOptionAction(function (Action $action) {
                            return $action->mutateFormDataUsing(function (array $data, Get $get) {
                                $data['test_id'] = $get('test_id');
                                return $data;
                            });
                        }),
                    TextInput::make('unit')
                        ->label('Unidad'),
                    Checkbox::make('is_calculated')
                        ->label('¿Es un componente calculado?')
                        ->reactive(),

                    Select::make('calculation_dependencies')
                        ->label('Campos necesarios para el cálculo')
                        ->multiple()
                        ->options(function (Get $get) {
                            $testId = $get('test_id');
                            return $testId
                                ? TestComponent::where('test_id', $testId)
                                ->where('is_group', false)
                                ->pluck('key', 'key')
                                : [];
                        })
                        ->visible(fn(Get $get) => $get('is_calculated')),

                    Textarea::make('calculation_formula')
                        ->label('Fórmula de cálculo')
                        ->helperText('Usa llaves para las variables, por ejemplo: ({hematocrito} * 10) / {rbc}')
                        ->visible(fn(Get $get) => $get('is_calculated')),
                ]),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('test.name')
                    ->label('Análisis')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('parent.name')
                    ->label('Grupo')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('name')
                    ->label('Nombre del componente')
                    ->searchable(),
                TextColumn::make('key')
                    ->label('Clave del componente')
                    ->searchable(),
                TextColumn::make('unit')
                    ->label('Unidad'),
                TextColumn::make('result_type')
                    ->label('Tipo de resultado'),
                TextInputColumn::make('order')
                    ->label('Orden')
                    ->rules(['required', 'max:10']),

            ])
            /*->defaultGroup(
                Group::make('parent.name')
                    ->orderQueryUsing(function (Builder $query, string $direction) {
                        return $query
                            ->orderBy('parent_id', $direction)
                            ->orderBy('order'); // o la columna que desees
                    })
                    ->titlePrefixedWithLabel(false),
            )*/
            //->defaultSort('order', 'asc')

            ->filters([
                SelectFilter::make('test_id')
                    ->label('Análisis')
                    ->relationship('test', 'name')
                    ->searchable()
                    ->preload(),

            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    //Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTestComponents::route('/'),
            'create' => Pages\CreateTestComponent::route('/create'),
            'edit' => Pages\EditTestComponent::route('/{record}/edit'),
        ];
    }
}
