<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TestResource\Pages;
use App\Filament\Resources\TestResource\RelationManagers;
use App\Models\Test;
use Filament\Forms;
use Filament\Forms\Components\Group;
use Filament\Forms\Form;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class TestResource extends Resource
{
    protected static ?string $model = Test::class;
    protected static ?string $navigationGroup = 'Laboratorio';
    protected static ?string $navigationLabel = 'Análisis';
    protected static ?string $pluralModelLabel = 'Analisis';
    protected static ?string $modelLabel = 'Análisis';
    protected static ?string $navigationIcon = 'heroicon-o-beaker';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                ->label('Nombre del Análisis')
                ->required()
                ->maxLength(255),
                Select::make('category_id')
                ->label('Categoría')
                ->relationship('category', 'name')
                ->required()
                ->createOptionForm([
                    TextInput::make('name')
                    ->label('Nombre de la Categoría')
                    ->required()
                    ->maxLength(255),
                ])
                ->preload()
                ->searchable(),
                Group::make()
                    ->columns(4)
                    ->columnSpanFull()
                    ->schema([
                        Toggle::make('is_active')
                            ->label('Estado del análisis')
                            ->default(true),
                        Toggle::make('is_profile')
                        ->label('El análisis es un perfil')
                        ->reactive()
                        ->default(false),
                        Toggle::make('is_grouped')
                        ->label('El análisis se imprimirá agrupado')
                        ->reactive()
                        ->default(false),
                        Toggle::make('page_break')
                        ->label('Imprimir seperado en una hoja')
                        ->reactive()
                        ->default(false),
                    ]),

                Select::make('profiles')
                    ->label('Agregar análisis al perfil')
                    ->relationship('profiles', 'name', function ($query) {
                        $query->where('is_profile', false); // solo tests normales
                    })
                    ->visible(fn($get) => $get('is_profile'))
                    ->multiple()
                    ->searchable()
                    ->preload()
                    ->columnSpanFull()
                    ->disabled(fn($get) => !$get('is_profile')),
                Repeater::make('prices')
                    ->label('Precios')
                    ->relationship('prices')
                    ->default([
                        [
                            'name' => 'particular',
                            'price' => 0,
                        ],
                        [
                            'name' => 'domicilio',
                            'price' => 0,
                        ],
                        [
                            'name' => 'seguro',
                            'price' => 0,
                        ],
                        [
                            'name' => 'emergencia',
                            'price' => 0,
                        ],
                    ])
                    ->schema([
                        TextInput::make('name')
                            ->label('Nombre')
                            ->required()
                            ->disabled()
                            ->dehydrated(),
                        TextInput::make('price')
                            ->label('Precio')
                            ->required()
                            ->numeric()
                            ->minValue(0),
                    ])
                    ->addable(false)
                    ->columns(2)
                    ->grid(4)
                    ->columnSpanFull()
                    ->deletable(false),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('Nombre del Análisis')
                    ->sortable()
                    ->searchable(),
                TextColumn::make('category.name')
                    ->label('Categoría')
                    ->sortable()
                    ->searchable(),
                TextColumn::make('prices')
                    ->label('Precios')
                    ->formatStateUsing(function ($record) {
                        return $record->prices
                            ->map(function ($price) {
                                return $price->price . ' Bs';
                            })
                            ->join(', ');
                    })
                    ->wrap(),
                ToggleColumn::make('is_active')
                    ->label('Activo')
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTests::route('/'),
            'create' => Pages\CreateTest::route('/create'),
            'edit' => Pages\EditTest::route('/{record}/edit'),
        ];
    }
}
